const fs = require("fs");
const path = require("path");
const { spawnSync } = require("child_process");

const BINARY_NAME = (process.platform === "win32") ? "abstract-cli.exe" : "abstract-cli";

function isAppleSilicon() {
  // uname and process.env.PROCESSOR_ARCHITECTURE will still report x86 when translated
  // see https://developer.apple.com/documentation/apple_silicon/about_the_rosetta_translation_environment#3616845
  const result = spawnSync("sysctl", ["sysctl.proc_translated"]);
  // node 15 would have `result.error`, but node 12 does not (has `result.status`)
  if (result.error || result.status === 1) {
    return false;
  }

  const strValue = result.stdout.toString();
  const isTranslated = strValue.includes("1");

  if (!isTranslated) {
    if (process.env.PROCESSOR_ARCHITECTURE && process.env.PROCESSOR_ARCHITECTURE !== "") {
      // TODO: once the desktop app supports arm64, double check
      //  this does what we think (it currently reports x86)
      return process.env.PROCESSOR_ARCHITECTURE === "arm64";
    }
  }

  return true;
}

function resolveBinDir() {
  let binDir = "bin";
  if (isAppleSilicon()) {
    binDir = path.join("bin", "arm64");
  }
  return path
    .resolve(__dirname, binDir)
    .replace(/[\\\/]app.asar[\\\/]/, `${path.sep}app.asar.unpacked${path.sep}`);
}

function resolveCLIBinary() {
  const binDir = resolveBinDir();
  if (process.platform === "darwin" || process.platform === "linux" || process.platform === "win32") {
    const binPath = path.join(binDir, BINARY_NAME);
    try {
      fs.accessSync(binPath, fs.constants.R_OK | fs.constants.X_OK);
      return binPath;
    } catch (error) {
      throw new Error(
        `${BINARY_NAME} could not be accessed at the expected path: ${binPath}`
      );
    }
  }
  throw new Error(
    `${BINARY_NAME} not supported on platform: ${process.platform}`
  );
}

module.exports = resolveCLIBinary();
